// ESC throttle Range Calibration for FreeFlight and other boards
// ==============================================================
// Works on standard KK board

/*
 * Setup:
 * 		Check/change PWM_PULSE defines
 *		Compile in AVR-Studio
 * 		Flash hex file to board
 * 		Connect up Rx & motors normally to Flight Controller (FC)
 * 		Remove props
 * 		Turn Tx on and put Collective High
 * 		Power on the FC.
 * 		after ESC signal bleeps, put Tx Collective to Low
 * 		Wait until FC LED starts flashing quickly (completed)
 * 		Power Off

 * FreeFlight board info
 * Measured PWM signals:
 *  900uS		Idle			dis-armed/armed @ min throttle
 * 1200uS		Step			armed throttle up
 * 1660uS		Maximum		maximum measured (including gyro)

 * LED status info
 * 		initial I'm alive flash
 *		Flashing every 0.5S : wait for Rx signal
 *		Off 								: doing ramp-down
 *		Rapid flashing			: finished! 
 */


#include <avr/io.h>  
#include <stdio.h>
#include <stdlib.h>
#include <util/delay.h>
#include <avr/interrupt.h> 
#include <stdbool.h>

#include "typedefs.h"
#include "io_cfg.h"

// These set the PWM throttle range for the ESC
#define PWM_PULSE_LOW 	1100			// nb: must be > 1000, else change Motor_Output()
#define PWM_PULSE_HIGH 	1700


volatile bool RxChannelsUpdatedFlag;

volatile uint16_t RxColl_ISR;						// ISR vars
uint16_t					RxColl_Start;					// ISR (private) vars
int16_t 					RxColl_In;						// 


/* ------------------------------------------- */
/* Function Prototypes */

void RxGetChannel( void );
void Motor_Output(uint16_t pulse);
void flashLED125mS(void);


// RX_COLL
ISR(INT1_vect)
{
	if ( RX_COLL )	
	{
		RxColl_Start = TCNT1;
	} else {			
		RxColl_ISR = TCNT1 - RxColl_Start;
		RxChannelsUpdatedFlag = true;
	}
}


/* ---------------
   main Processing
	 --------------- */
int main(void)
{
	int16_t RxColl_Initial;
	uint16_t ESC_Pulse;


	MCUCR |= (1<<PUD);	// Pull-up Disable


	RX_COLL_DIR   		= INPUT;

	M1_DIR 					 	= OUTPUT;
	M2_DIR 				 		= OUTPUT;
	M3_DIR 			 			= OUTPUT;
	M4_DIR 			 			= OUTPUT;

	LED_DIR 			 		= OUTPUT;

	LED 							= 0;

	// external interrupts
	EICRA  = (1 << ISC10);		// Any change INT1
	EIMSK  = (1 << INT1);			// External Interrupt Mask Register
	EIFR   = (1 << INTF1);

	// timer0 (8bit) - run @ 8MHz
	// used to control ESC/servo pulse length
	TCCR0A = 0;						// normal operation
	TCCR0B = (1 << CS00);	// clk/0
	TIMSK0 = 0; 					// no interrupts

	// timer1 (16bit) - run @ 1Mhz
	// used to measure Rx Signals & control ESC/servo output rate
	TCCR1A = 0;
	TCCR1B = (1 << CS11);


	RxChannelsUpdatedFlag = false;

	RxColl_ISR 		= 0;		// init channels 

	sei();								// Global Interrupts 

	flashLED125mS();


	// 1.5 second delay
	_delay_ms(1500);


	// Wait for Rx Collective to go high
	do
	{
		_delay_ms(250);
		flashLED125mS();
		RxGetChannel();
	}	while (RxColl_In < 160);



	// now do ESC throttle calibration
	// -------------------------------
	LED	= 0;

	// save initial
	RxColl_Initial = RxColl_In;

	// output Max pulse until user lowers TxColl
	while ((RxColl_Initial - RxColl_In) < 5)
	{
		Motor_Output( PWM_PULSE_HIGH );
		_delay_ms(18);
		RxGetChannel();
	}

	// Disable Rx Input since we don't need it!
	EIMSK = 0;

	// now do ramp-down to Min
	ESC_Pulse = PWM_PULSE_HIGH;

	while ( ESC_Pulse > PWM_PULSE_LOW )
	{
		Motor_Output(ESC_Pulse);
		_delay_ms(18);

		ESC_Pulse -= 5;
	}

	// now loop 4-ever!
	while (1)
	{
		Motor_Output( PWM_PULSE_LOW );
		_delay_ms(18);

		LED = (LED ? 0 : 1);
	}

	return 1;
}


void RxGetChannel( void )
{
	uint16_t RxTemp;

  do {
	  RxChannelsUpdatedFlag = false;
		RxTemp = RxColl_ISR;
	} while (	RxChannelsUpdatedFlag );

	RxColl_In = RxTemp - 1100;
	RxColl_In = (RxColl_In >> 2);			// 0 -> 220
}



/* ------------------------
	Ouput PWM signal to ESC's
  ------------------------- */
// pulse 1000 -> 2000
void Motor_Output(uint16_t pulse)
{
	uint8_t i;
	uint16_t tcnt1, elapsed;

	pulse -= 1000;	// remove the fixed portion
	// now 0 to 1000
	pulse /= 4;			// convert to 4uS increments

	tcnt1 = TCNT1;

	// turn on pins
	M1 = 1;
	M2 = 1;
	M3 = 1;
	M4 = 1;

	// wait ~1mS
	do
	{
		nop();	nop();  nop();  nop();
		elapsed = TCNT1 - tcnt1;
	} while (elapsed < 980);


	for (i=0;i<=250;i++)	// motors 0->250, 1120->1920 uS
	{
    if (i==pulse)
		{
			M1 = 0;
			M2 = 0;
			M3 = 0;
			M4 = 0;
		}

		while (TCNT0 < 30)		// 4uS @ 8MHz = 32
		{
			nop();	nop();  nop();  nop();
		}
		TCNT0 -= 30;
	}
}



// Flash LED for 125mS
void flashLED125mS(void)
{
	_delay_ms(125);
	// flash LED
	LED = 1;
	_delay_ms(125);
	LED = 0;
}





