.equ HottBinaryMode =0x80
.equ HottEAM =0x8E // Electric Air Module ID
.equ HottElectricalAirSensorID =0xE0 // Electric Air Sensor ID
.equ HottNotificationUndervoltage =0x10

.macro HottReceiverMode
	lds t, UCSR1B
	clt
	bld t, TXEN1
	set
	bld t, RXEN1
	sts UCSR1B, t
.endmacro

.macro HottTransmitterMode
	lds tt, UCSR1B
	clt
	bld tt, RXEN1
	set
	bld tt, TXEN1
	sts UCSR1B, tt
.endmacro

HottInit:
	;init buffer
	ldx RamHottTelemetry
	ldz HottTelemetryData * 2
	ldi t, 44
HottInitBufferLoop:
	lpm yl, z+
	st x+, yl
	dec t
	brne HottInitBufferLoop

	;set pointer out of buffer -> means nothing to transmit
	ldx RamHottTelemetry
	adiw x, 45
	sts HottTxPointerL, xl
	sts HottTxPointerH, xh

	; Set baud rate to 19200
	clr t
	sts UBRR1H, t
	ldi t, 64
	sts UBRR1L, t

	; Set frame format: 8data, 1stop bit
	ldi t, (3<<UCSZ10)
	sts UCSR1C, t

	; Disable Transmitter, Enable Receiver and RX Complete Interrupt
	lds t, UCSR1B
	clt
	bld t, TXEN1
	set
	bld t, RXCIE1
	bld t, RXEN1
	sts UCSR1B, t

	ret

HottUpdate:
	rvbrflagfalse flagHottOn, HottDisabled
	rvbrflagfalse flagCppmOn, HottDisabled
	call HottUpdateBattery
	call HottUpdateFlightTime
	ret

HottDisabled:
	ret

IsrHottReceive:
	in SregSaver, sreg

	push xl
	push xh

	lds xl, HottRxLastByte

	; Get received data from buffer
	load tt, UDR1
	sts HottRxLastByte, tt

	cpi tt, HottEAM
	brne IsrHottReceiveEnd

	cpi xl, HottBinaryMode
	breq HottStartTransmit

IsrHottReceiveEnd:
	pop xh
	pop xl

	out sreg, SregSaver
	reti

HottStartTransmit:
	HottTransmitterMode

	;Set pointer to before buffer (for delay) and clear checksum
	ldx RamHottTelemetry
	sbiw x, 2
	sts HottTxPointerL, xl
	sts HottTxPointerH, xh
	clr xl
	sts HottTxChecksum, xl

	pop xh
	pop xl

	out sreg, SregSaver
	reti

HottPoll:
	rvbrflagfalse flagHottOn, HottDisabled
	rvbrflagfalse flagCppmOn, HottDisabled

	;pointer before buffer? yes: means transmit delay
	ldy RamHottTelemetry
	lds xl, HottTxPointerL
	lds xh, HottTxPointerH
	cp xl, yl
	cpc xh, yh
	brlo HottTransmitDelay

	;pointer at end of data? no: transmit next byte
	adiw y, 44
	cp xl, yl
	cpc xh, yh
	brlo HottTransmitNextByte

	;pointer out of buffer? yes: no data to send, return
	adiw y, 1
	cp xl, yl
	cpc xh, yh
	brge HottTransmitNoData

	;end of buffer reached, transmit checksum
	lds t, HottTxChecksum
	store UDR1, t
	st x+, t
	sts HottTxPointerL, xl
	sts HottTxPointerH, xh

	ret

HottTransmitDelay:
	adiw x, 1
	sts HottTxPointerL, xl
	sts HottTxPointerH, xh
	ret

HottTransmitNoData:
	HottReceiverMode
	ret

HottTransmitNextByte:
	ld t, x+
	store UDR1, t
	sts HottTxPointerL, xl
	sts HottTxPointerH, xh
	lds yl, HottTxChecksum
	add yl, t
	sts HottTxChecksum, yl

	ret

HottUpdateBattery:
	ldz RamHottTelemetry

	;check: BatteryVoltageLowpass instead of raw voltage?
	b16ldi Temp, 68.75067139
	b16mul Temp, BatteryVoltage, Temp
	b16fdiv Temp, 8
	b16load Temp
	movw y, z
	adiw y, 30
	st y, xl

	rvbrflagfalse flagLvaBuzzerOn, HottNoBatAlarm

	;when low voltage alarm is triggered, also send a notification
	movw y, z
	adiw y, 2
	ldi t, HottNotificationUndervoltage
	st y, t

	;invert voltage in mikrokopter view
	movw y, z
	adiw y, 4
	ldi t, 0x80
	st y, t

	ret

HottNoBatAlarm:
	;clear notification
	movw y, z
	adiw y, 2
	clr t
	st y, t

	;restore view
	movw y, z
	adiw y, 4
	st y, t

	ret

HottUpdateFlightTime:
	ldz RamHottTelemetry
	lds xh, FlightTimeMinutes
	lds xl, FlightTimeSeconds

	movw y, z
	adiw y, 39
	st y, xh

	movw y, z
	adiw y, 40
	st y, xl

	ret


HottTelemetryData:
.db	  0x7C,	\
	  HottEAM,	\
	  0x00, /* Alarm */	\
	  HottElectricalAirSensorID,	\
	  0x00, 0x00, /* Alarm Value 1 and 2 */	\
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* Low Voltage Cell 1-7 in 2mV steps */	\
	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* High Voltage Cell 1-7 in 2mV steps */ \
	  0x00, 0x00, /* Battetry 1 LSB/MSB in 100mv steps, 50 == 5V */ \
	  0x00, 0x00, /* Battetry 2 LSB/MSB in 100mv steps, 50 == 5V */ \
	  0x14, /* Temp 1, Offset of 20. 20 == 0C */ \
	  0x14, /* Temp 2, Offset of 20. 20 == 0C */ \
	  0xF4, 0x01, /* Height. Offset -500. 500 == 0 */ \
	  0x00, 0x00, /* Current LSB, MSB 1 = 0.1A */ \
	  0x00, 0x00, /* Drive Voltage */ \
	  0x00, 0x00, /* mAh */ \
	  0x78, 0x00, /* m/s */ \
	  0x78, /* m/3s */ \
	  0x00, 0x00, /* RPM. 10er steps, 300 == 3000rpm */ \
	  0x00, /* Electric minutes */ \
	  0x00, /* Electric seconds */ \
	  0x00, /* Speed */ \
	  0x00, /* Version Number */ \
	  0x7D /* End sign */
;	  0x00, /* Checksum */


